import type {WalletContextState} from '@solana/wallet-adapter-react';
import {CONFIG, Phase} from '../config';
type Action='CREATE'|'CLAIM';
function rolePrefix(phase:Phase){return phase==='AMBASSADOR'?'amb':phase==='MERCHANT'?'merch':'cons';}
export function subdomainForMint(phase:Phase, nftMint:string){
  const prefix=rolePrefix(phase);
  const mintPrefix=nftMint.replace(/[^A-Za-z0-9]/g,'').slice(0,6);
  return `${prefix}-${mintPrefix}.opentill.sol`;
}
async function getNonce(action:Action,walletPk:string){
  const res=await fetch(`${CONFIG.apiBaseUrl}/api/nonce`,{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({action,wallet:walletPk})});
  if(!res.ok) throw new Error(await res.text());
  return res.json() as Promise<{nonce:string;expiresAt:number}>;
}
async function signMessage(wallet:WalletContextState,message:string){
  if(!wallet.signMessage) throw new Error('Wallet does not support message signing.');
  const encoded=new TextEncoder().encode(message);
  const sigBytes=await wallet.signMessage(encoded);
  return btoa(String.fromCharCode(...sigBytes));
}
export async function createSubdomain(params:{wallet:WalletContextState;phase:Phase;nftMint:string;}){
  const {wallet,phase,nftMint}=params;
  if(!wallet.publicKey) throw new Error('Wallet not connected.');
  const walletPk=wallet.publicKey.toBase58();
  const {nonce}=await getNonce('CREATE',walletPk);
  const subdomain=subdomainForMint(phase,nftMint);
  const message=`openTILL create: ${subdomain} for mint ${nftMint} nonce ${nonce}`;
  const signature=await signMessage(wallet,message);
  const res=await fetch(`${CONFIG.apiBaseUrl}/api/subdomain/create`,{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({wallet:walletPk,phase,nftMint,subdomain,nonce,message,signature})});
  if(!res.ok) throw new Error(await res.text());
  return res.json() as Promise<{subdomain:string}>;
}
export async function claimSubdomain(params:{wallet:WalletContextState;phase:Phase;nftMint:string;}){
  const {wallet,phase,nftMint}=params;
  if(!wallet.publicKey) throw new Error('Wallet not connected.');
  const walletPk=wallet.publicKey.toBase58();
  const {nonce}=await getNonce('CLAIM',walletPk);
  const subdomain=subdomainForMint(phase,nftMint);
  const message=`openTILL claim: ${subdomain} for mint ${nftMint} nonce ${nonce}`;
  const signature=await signMessage(wallet,message);
  const res=await fetch(`${CONFIG.apiBaseUrl}/api/subdomain/claim`,{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({wallet:walletPk,phase,nftMint,subdomain,nonce,message,signature})});
  if(!res.ok) throw new Error(await res.text());
  return res.json() as Promise<{subdomain:string}>;
}
